import 'dart:developer';

import 'package:easy_localization/easy_localization.dart';
import 'package:emartstore/constants.dart';
import 'package:emartstore/main.dart';
import 'package:emartstore/model/User.dart';
import 'package:emartstore/services/FirebaseHelper.dart';
import 'package:emartstore/services/helper.dart';
import 'package:emartstore/services/show_toast_dailog.dart';
import 'package:emartstore/ui/container/ContainerScreen.dart';
import 'package:emartstore/ui/phoneAuth/PhoneNumberInputScreen.dart';
import 'package:emartstore/ui/resetPasswordScreen/ResetPasswordScreen.dart';
import 'package:emartstore/ui/subscription_screen/app_not_access_screen.dart';
import 'package:emartstore/ui/subscription_screen/subscription_screens.dart';
import 'package:flutter/material.dart';
import 'package:the_apple_sign_in/the_apple_sign_in.dart' as apple;

class LoginScreen extends StatefulWidget {
  @override
  State createState() {
    return _LoginScreen();
  }
}

class _LoginScreen extends State<LoginScreen> {
  TextEditingController _emailController = TextEditingController();
  TextEditingController _passwordController = TextEditingController();
  AutovalidateMode _validate = AutovalidateMode.disabled;
  GlobalKey<FormState> _key = GlobalKey();

  @override
  Widget build(BuildContext context) {
    return Scaffold(
      appBar: AppBar(
        backgroundColor: Colors.transparent,
        iconTheme: IconThemeData(
            color: isDarkMode(context) ? Colors.white : Colors.black),
        elevation: 0.0,
      ),
      body: Form(
        key: _key,
        autovalidateMode: _validate,
        child: Padding(
          padding: const EdgeInsets.symmetric(horizontal: 16),
          child: ListView(
            children: <Widget>[
              Padding(
                padding: const EdgeInsets.only(top: 32.0, bottom: 30),
                child: Text(
                  'Log In'.tr(),
                  style: TextStyle(
                      color: Color(COLOR_PRIMARY),
                      fontSize: 25.0,
                      fontWeight: FontWeight.bold),
                ),
              ),

              /// email address text field, visible when logging with email
              /// and password
              TextFormField(
                  textAlignVertical: TextAlignVertical.center,
                  textInputAction: TextInputAction.next,
                  validator: validateEmail,
                  controller: _emailController,
                  style: TextStyle(fontSize: 18.0),
                  keyboardType: TextInputType.emailAddress,
                  cursorColor: Color(COLOR_PRIMARY),
                  decoration: InputDecoration(
                    contentPadding: EdgeInsets.only(left: 16, right: 16),
                    hintText: 'Email Address'.tr(),
                    focusedBorder: OutlineInputBorder(
                        borderRadius: BorderRadius.circular(25.0),
                        borderSide: BorderSide(
                            color: Color(COLOR_PRIMARY), width: 2.0)),
                    errorBorder: OutlineInputBorder(
                      borderSide: BorderSide(
                          color: Theme.of(context).colorScheme.error),
                      borderRadius: BorderRadius.circular(25.0),
                    ),
                    focusedErrorBorder: OutlineInputBorder(
                      borderSide: BorderSide(
                          color: Theme.of(context).colorScheme.error),
                      borderRadius: BorderRadius.circular(25.0),
                    ),
                    enabledBorder: OutlineInputBorder(
                      borderSide: BorderSide(color: Colors.grey.shade200),
                      borderRadius: BorderRadius.circular(25.0),
                    ),
                  )),

              /// password text field, visible when logging with email and
              /// password
              Padding(
                padding: const EdgeInsets.only(top: 14.0),
                child: TextFormField(
                    textAlignVertical: TextAlignVertical.center,
                    controller: _passwordController,
                    obscureText: true,
                    validator: validatePassword,
                    onFieldSubmitted: (password) => _login(),
                    textInputAction: TextInputAction.done,
                    style: TextStyle(fontSize: 18.0),
                    cursorColor: Color(COLOR_PRIMARY),
                    decoration: InputDecoration(
                      contentPadding: EdgeInsets.only(left: 16, right: 16),
                      hintText: 'Password'.tr(),
                      focusedBorder: OutlineInputBorder(
                          borderRadius: BorderRadius.circular(25.0),
                          borderSide: BorderSide(
                              color: Color(COLOR_PRIMARY), width: 2.0)),
                      errorBorder: OutlineInputBorder(
                        borderSide: BorderSide(
                            color: Theme.of(context).colorScheme.error),
                        borderRadius: BorderRadius.circular(25.0),
                      ),
                      focusedErrorBorder: OutlineInputBorder(
                        borderSide: BorderSide(
                            color: Theme.of(context).colorScheme.error),
                        borderRadius: BorderRadius.circular(25.0),
                      ),
                      enabledBorder: OutlineInputBorder(
                        borderSide: BorderSide(color: Colors.grey.shade200),
                        borderRadius: BorderRadius.circular(25.0),
                      ),
                    )),
              ),

              /// forgot password text, navigates user to ResetPasswordScreen
              /// and this is only visible when logging with email and password
              Padding(
                padding: const EdgeInsets.only(bottom: 30, top: 10),
                child: Align(
                  alignment: Alignment.centerRight,
                  child: GestureDetector(
                    onTap: () => push(context, ResetPasswordScreen()),
                    child: Text(
                      'Forgot password?'.tr(),
                      style: TextStyle(
                          color: Colors.lightBlue,
                          fontWeight: FontWeight.bold,
                          fontSize: 15,
                          letterSpacing: 1),
                    ),
                  ),
                ),
              ),

              /// the main action button of the screen, this is hidden if we
              /// received the code from firebase
              /// the action and the title is base on the state,
              /// * logging with email and password: send email and password to
              /// firebase
              /// * logging with phone number: submits the phone number to
              /// firebase and await for code verification
              Padding(
                padding: const EdgeInsets.symmetric(horizontal: 50),
                child: ElevatedButton(
                  style: ElevatedButton.styleFrom(
                    backgroundColor: Color(COLOR_PRIMARY),
                    padding: EdgeInsets.only(top: 8, bottom: 8),
                    shape: RoundedRectangleBorder(
                      borderRadius: BorderRadius.circular(25.0),
                      side: BorderSide(
                        color: Color(COLOR_PRIMARY),
                      ),
                    ),
                  ),
                  child: Text(
                    'Log In'.tr(),
                    style: TextStyle(
                      fontWeight: FontWeight.bold,
                      color: isDarkMode(context) ? Colors.black : Colors.white,
                    ),
                  ),
                  onPressed: () => _login(),
                ),
              ),
              Padding(
                padding: const EdgeInsets.all(32.0),
                child: Center(
                  child: Text(
                    'OR'.tr(),
                    style: TextStyle(
                        color:
                            isDarkMode(context) ? Colors.white : Colors.black),
                  ).tr(),
                ),
              ),

              FutureBuilder<bool>(
                future: apple.TheAppleSignIn.isAvailable(),
                builder: (context, snapshot) {
                  if (snapshot.connectionState == ConnectionState.waiting) {
                    return CircularProgressIndicator.adaptive(
                      valueColor: AlwaysStoppedAnimation(
                        Color(COLOR_PRIMARY),
                      ),
                    );
                  }
                  if (!snapshot.hasData || (snapshot.data != true)) {
                    return Container();
                  } else {
                    return Padding(
                      padding: const EdgeInsets.only(
                          right: 40.0, left: 40.0, bottom: 20),
                      child: apple.AppleSignInButton(
                        cornerRadius: 25.0,
                        type: apple.ButtonType.signIn,
                        style: isDarkMode(context)
                            ? apple.ButtonStyle.white
                            : apple.ButtonStyle.black,
                        onPressed: () => loginWithApple(),
                      ),
                    );
                  }
                },
              ),

              SizedBox(
                height: 30,
              ),

              /// switch between login with phone number and email login states
              InkWell(
                onTap: () {
                  push(context, PhoneNumberInputScreen(login: true));
                },
                child: Padding(
                  padding: const EdgeInsets.symmetric(horizontal: 10),
                  child: Container(
                    margin: EdgeInsets.symmetric(horizontal: 40),
                    decoration: BoxDecoration(
                        borderRadius: BorderRadius.circular(30),
                        color: Colors.white,
                        border:
                            Border.all(color: Color(COLOR_PRIMARY), width: 1)),
                    child: Padding(
                      padding: EdgeInsets.all(8.0),
                      child: Row(
                        mainAxisAlignment: MainAxisAlignment.center,
                        children: [
                          Icon(
                            Icons.call,
                            color: Color(COLOR_PRIMARY),
                          ),
                          SizedBox(
                            width: 5,
                          ),
                          Text(
                            'Login with phone number'.tr(),
                            style: TextStyle(
                                color: Color(COLOR_PRIMARY),
                                fontWeight: FontWeight.bold,
                                fontSize: 15,
                                letterSpacing: 1),
                          ),
                        ],
                      ),
                    ),
                  ),
                ),
              )
            ],
          ),
        ),
      ),
    );
  }

  _login() async {
    if (_key.currentState?.validate() ?? false) {
      _key.currentState!.save();
      await _loginWithEmailAndPassword(
          _emailController.text.trim(), _passwordController.text.trim());
    } else {
      setState(() {
        _validate = AutovalidateMode.onUserInteraction;
      });
    }
  }

  /// login with email and password with firebase
  /// @param email user email
  /// @param password user password
  _loginWithEmailAndPassword(String email, String password) async {
    ShowToastDialog.showLoader('Logging in, please wait...'.tr());
    dynamic result = await FireStoreUtils.loginWithEmailAndPassword(
        email.trim(), password.trim());
    await ShowToastDialog.closeLoader();
    if (result != null && result is User && result.role == 'vendor') {
      if (result.active == true) {
        await FireStoreUtils.updateCurrentUser(result);
        print("result ans:" + result.fcmToken);
        MyAppState.currentUser = result;
        if (result.vendorID.isNotEmpty) {
          await FireStoreUtils.firestore
              .collection(VENDORS)
              .doc(result.vendorID)
              .update({"fcmToken": result.fcmToken});
        }

        if (MyAppState.currentUser?.section_id.isNotEmpty == true) {
          await FireStoreUtils.getSectionsById(
                  MyAppState.currentUser!.section_id)
              .then(
            (value) {
              if (value != null) {
                selectedSectionModel = value;
              }
            },
          );
        }

        if (MyAppState.currentUser?.section_id.isEmpty == true &&
            isSubscriptionModelApplied == false) {
          pushAndRemoveUntil(context, ContainerScreen(user: result), false);
        } else if (result.subscriptionPlanId == null ||
            isExpire(result) == true) {
          if ((selectedSectionModel != null &&
                  selectedSectionModel!.adminCommision!.enable == false) &&
              isSubscriptionModelApplied == false) {
            pushAndRemoveUntil(context, ContainerScreen(user: result), false);
          } else {
            pushAndRemoveUntil(
                context,
                SubscriptionScreens(
                  isShowAppBar: false,
                  isDropDownDisble:
                      result.section_id.isEmpty == true ? false : true,
                ),
                false);
          }
        } else if (result.subscriptionPlan?.features?.ownerMobileApp == true) {
          pushAndRemoveUntil(context, ContainerScreen(user: result), false);
        } else {
          pushAndRemoveUntil(context, AppNotAccessScreen(), false);
        }
      } else {
        showAlertDialog(
            context,
            'Your account has been disabled, Please contact to admin.'.tr(),
            "",
            true);
      }
    } else if (result != null) {
      showAlertDialog(context, "Couldn't Authenticate".tr(), result, true);
    }
  }

  ///dispose text editing controllers to avoid memory leaks
  @override
  void dispose() {
    _emailController.dispose();
    _passwordController.dispose();
    super.dispose();
  }

  loginWithApple() async {
    try {
      ShowToastDialog.showLoader('Logging in, Please wait...'.tr());
      dynamic result = await FireStoreUtils.loginWithApple();
      ShowToastDialog.closeLoader();
      if (result != null && result is User) {
        MyAppState.currentUser = result;
        if (MyAppState.currentUser!.active == true) {
          if (MyAppState.currentUser!.section_id.isNotEmpty) {
            await FireStoreUtils.getSectionsById(
                    MyAppState.currentUser!.section_id)
                .then(
              (value) {
                if (value != null) {
                  selectedSectionModel = value;
                }
              },
            );
          }

          if (result.subscriptionPlanId == null || isExpire(result) == true) {
            if ((selectedSectionModel != null &&
                    selectedSectionModel!.adminCommision!.enable == false) &&
                isSubscriptionModelApplied == false) {
              pushAndRemoveUntil(context, ContainerScreen(user: result), false);
            } else {
              pushAndRemoveUntil(
                  context,
                  SubscriptionScreens(
                    isShowAppBar: false,
                    isDropDownDisble:
                        result.section_id.isEmpty == true ? false : true,
                  ),
                  false);
            }
          } else if (result.subscriptionPlan?.features?.ownerMobileApp ==
              true) {
            pushAndRemoveUntil(context, ContainerScreen(user: result), false);
          } else {
            pushAndRemoveUntil(context, AppNotAccessScreen(), false);
          }
        } else {
          showAlertDialog(
              context,
              'Your account has been disabled, Please contact to admin.'.tr(),
              "",
              true);
        }
      } else if (result != null && result is String) {
        showAlertDialog(context, 'Error'.tr(), result.tr(), true);
      } else {
        showAlertDialog(
            context, 'Error'.tr(), "Couldn't login with apple.".tr(), true);
      }
    } catch (e, s) {
      ShowToastDialog.closeLoader();
      print('_LoginScreen.loginWithApple $e $s');
      showAlertDialog(
          context, 'Error'.tr(), "Couldn't login with apple.".tr(), true);
    }
  }
}
